# user_library.py
#
# This is a custom user library that extends the main Library.py.
# It provides a complete set of simple functions for creating primitive objects,
# arranging them, and building procedural structures.

import bpy
import math
from mathutils import Vector, Color

# --- CRITICAL FIX ---
# Import the entire base library so this file can access its functions.
# All calls to base functions must now use the `Library.` prefix.
import Library

print("--- User Library with Color Primitives loaded successfully! ---")

##########################################################################
# Theme: Color Utilities
# Helper function to apply color to an object.
##########################################################################

def Set_Object_Color(obj_ref, color=(1.0, 1.0, 1.0), name=""):
    """
    Sets the color of an object by creating and assigning a new material.

    This is a helper function used by the Create_* functions when a color
    is provided. It finds the main shader and sets its base color.
    """
    obj = Library.Get_Object_Any(obj_ref)
    if not obj:
        print(f"Error: Could not find object '{obj_ref}' to color.")
        return None
        
    mat_name = name if name else f"{obj.name}_ColorMat"
    mat = Library.Mat_Create(mat_name, obj)
    
    principled_bsdf = mat.node_tree.nodes.get("Principled BSDF")
    if principled_bsdf:
        principled_bsdf.inputs["Base Color"].default_value = (*color, 1.0) # (R, G, B, Alpha)
        
    return mat

##########################################################################
# Theme: A Complete Set of Primitives with Integrated Color
##########################################################################

def Create_Cube(name="Cube", location=(0, 0, 0), size=2.0, color=None):
    """
    Creates a new Cube mesh object with an optional color.
    Parameters: name (str), location (tuple), size (float), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Cube object.
    Example Usage: >>> red_cube = Create_Cube("RedBlock", color=(1, 0, 0))
    """
    bpy.ops.mesh.primitive_cube_add(size=size, location=location)
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Cube '{name}'.")
    return new_obj

def Create_Plane(name="Plane", location=(0, 0, 0), size=2.0, color=None):
    """
    Creates a new Plane mesh object with an optional color.
    Parameters: name (str), location (tuple), size (float), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Plane object.
    Example Usage: >>> floor = Create_Plane("Floor", size=20, color=(0.1, 0.1, 0.1))
    """
    bpy.ops.mesh.primitive_plane_add(size=size, location=location)
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Plane '{name}'.")
    return new_obj

def Create_Sphere(name="Sphere", location=(0, 0, 0), radius=1.0, segments=32, rings=16, color=None):
    """
    Creates a new UV Sphere mesh object with an optional color.
    Parameters: name (str), location (tuple), radius (float), segments (int), rings (int), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Sphere object.
    Example Usage: >>> ball = Create_Sphere("Ball", radius=0.5, color=(0, 0.8, 0))
    """
    bpy.ops.mesh.primitive_uv_sphere_add(
        radius=radius, segments=segments, ring_count=rings, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Sphere '{name}'.")
    return new_obj

def Create_Ico_Sphere(name="IcoSphere", location=(0, 0, 0), radius=1.0, subdivisions=2, color=None):
    """
    Creates a new Ico Sphere (geodesic) mesh object with an optional color.
    Parameters: name (str), location (tuple), radius (float), subdivisions (int), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Ico Sphere object.
    Example Usage: >>> golf_ball = Create_Ico_Sphere("GolfBall", subdivisions=4, color=(0.9, 0.9, 0.9))
    """
    bpy.ops.mesh.primitive_ico_sphere_add(
        radius=radius, subdivisions=subdivisions, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Ico Sphere '{name}'.")
    return new_obj

def Create_Cylinder(name="Cylinder", location=(0, 0, 0), radius=1.0, height=2.0, vertices=32, color=None):
    """
    Creates a new Cylinder mesh object with an optional color.
    Parameters: name (str), location (tuple), radius (float), height (float), vertices (int), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Cylinder object.
    Example Usage: >>> pipe = Create_Cylinder("Pipe", height=10, color=(0.5, 0.5, 0.5))
    """
    bpy.ops.mesh.primitive_cylinder_add(
        radius=radius, depth=height, vertices=vertices, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Cylinder '{name}'.")
    return new_obj

def Create_Cone(name="Cone", location=(0, 0, 0), radius=1.0, height=2.0, vertices=32, color=None):
    """
    Creates a new Cone mesh object with an optional color.
    Parameters: name (str), location (tuple), radius (float), height (float), vertices (int), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Cone object.
    Example Usage: >>> traffic_cone = Create_Cone("WarningCone", color=(1, 0.3, 0))
    """
    bpy.ops.mesh.primitive_cone_add(
        radius1=radius, depth=height, vertices=vertices, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Cone '{name}'.")
    return new_obj

def Create_Torus(name="Torus", location=(0, 0, 0), major_radius=1.0, minor_radius=0.25, color=None):
    """
    Creates a new Torus (donut) mesh object with an optional color.
    Parameters: name (str), location (tuple), major_radius (float), minor_radius (float), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Torus object.
    Example Usage: >>> ring = Create_Torus("Ring", minor_radius=0.1, color=(0.9, 0.7, 0.1))
    """
    bpy.ops.mesh.primitive_torus_add(
        major_radius=major_radius, minor_radius=minor_radius,
        major_segments=48, minor_segments=12, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Torus '{name}'.")
    return new_obj

def Create_Grid(name="Grid", location=(0, 0, 0), size=2.0, subdivisions=10, color=None):
    """
    Creates a new subdivided Grid mesh object with an optional color.
    Parameters: name (str), location (tuple), size (float), subdivisions (int), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Grid object.
    Example Usage: >>> net = Create_Grid("Net", subdivisions=20, color=(0.8, 0.8, 0.8))
    """
    bpy.ops.mesh.primitive_grid_add(
        size=size, x_subdivisions=subdivisions, y_subdivisions=subdivisions, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Grid '{name}'.")
    return new_obj

def Create_Circle(name="Circle", location=(0, 0, 0), radius=1.0, vertices=32, fill=True, color=None):
    """
    Creates a new Circle mesh object with an optional color.
    Parameters: name (str), location (tuple), radius (float), vertices (int), fill (bool), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Circle object.
    Example Usage: >>> disc = Create_Circle("Disc", fill=True, color=(0, 0.5, 1))
    """
    fill_type = 'NGON' if fill else 'NOTHING'
    bpy.ops.mesh.primitive_circle_add(
        radius=radius, vertices=vertices, fill_type=fill_type, location=location
    )
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Circle '{name}'.")
    return new_obj

def Create_Monkey(name="Suzanne", location=(0, 0, 0), size=2.0, color=None):
    """
    Creates a new Monkey (Suzanne) mesh object with an optional color.
    Parameters: name (str), location (tuple), size (float), color (tuple, optional).
    Returns: bpy.types.Object - The newly created Monkey object.
    Example Usage: >>> gold_monkey = Create_Monkey(color=(0.9, 0.7, 0.1))
    """
    bpy.ops.mesh.primitive_monkey_add(size=size, location=location)
    new_obj = bpy.context.active_object
    Library.Rename_Object(new_obj, name)
    if color is not None:
        Set_Object_Color(new_obj, color)
    print(f"Created Monkey '{name}'.")
    return new_obj
    

##########################################################################
# Theme: Object Sorting and Ordering
##########################################################################

def Sort_Selected_By_Size(reverse=False):
    """
    Sorts the selection order by object size (volume), setting the active object.
    Parameters: reverse (bool) - If True, sorts largest to smallest, making the smallest active.
    Returns: list[bpy.types.Object] - The list of objects in their new sorted order.
    Example Usage: >>> Sort_Selected_By_Size() # Makes the largest object active
    """
    selected_objects = bpy.context.selected_objects
    if len(selected_objects) < 2:
        print("Warning: At least two objects must be selected to sort.")
        return []

    sorted_objects = sorted(
        selected_objects,
        key=lambda obj: obj.dimensions.x * obj.dimensions.y * obj.dimensions.z,
        reverse=not reverse
    )

    Library.Deselect_All()
    for obj in sorted_objects:
        Library.Select_Object(obj)

    Library.Set_Active_Object(sorted_objects[-1])
    
    result_order = "smallest to largest" if not reverse else "largest to smallest"
    print(f"Selection sorted from {result_order}. '{sorted_objects[-1].name}' is now the active object.")
    return sorted_objects

##########################################################################
# Theme: Object Alignment
##########################################################################

def Align_Selected_Objects(axis='Z'):
    """
    Aligns all selected objects to the location of the active object on a specific axis.
    Parameters: axis (str) - The axis to align on ('X', 'Y', or 'Z').
    Returns: None.
    Example Usage: >>> Align_Selected_Objects(axis='Z') # Makes all objects level with the active one
    """
    selected_objects = bpy.context.selected_objects
    active_obj = Library.Get_Active_Object()
    if len(selected_objects) < 2 or not active_obj:
        print("Warning: Select at least two objects, with one being active, to align.")
        return

    axis_map = {'X': 0, 'Y': 1, 'Z': 2}
    if axis.upper() not in axis_map:
        print(f"Error: Invalid axis '{axis}'. Use 'X', 'Y', or 'Z'.")
        return
        
    axis_index = axis_map[axis.upper()]
    target_location = active_obj.location[axis_index]

    for obj in selected_objects:
        if obj != active_obj:
            new_loc = list(obj.location)
            new_loc[axis_index] = target_location
            Library.Trans_Location(obj, new_loc)
            
    print(f"Aligned {len(selected_objects)} objects to the active object on the {axis}-axis.")

##########################################################################
# Theme: Object Stacking and Spacing
##########################################################################

def _get_object_dimension(obj, axis_index):
    """Internal helper to get an object's world dimension along an axis."""
    return obj.dimensions[axis_index]

def Stack_Selected_Objects(axis='X', gap=0.0):
    """
    Stacks selected objects side-by-side along an axis based on their bounding boxes.
    Parameters: axis (str) - Stacking axis ('X','Y','Z'), gap (float) - The space between objects.
    Returns: None.
    Example Usage: >>> Stack_Selected_Objects(axis='X', gap=0.1) # Lines up objects along X with a small gap
    """
    selected_objects = bpy.context.selected_objects
    if len(selected_objects) < 2:
        print("Warning: At least two objects must be selected to stack.")
        return

    axis_map = {'X': 0, 'Y': 1, 'Z': 2}
    axis_index = axis_map[axis.upper()]

    sorted_objects = sorted(selected_objects, key=lambda obj: obj.location[axis_index])
    cursor = sorted_objects[0].location[axis_index] - (_get_object_dimension(sorted_objects[0], axis_index) / 2)

    for obj in sorted_objects:
        obj_dim = _get_object_dimension(obj, axis_index)
        new_pos = cursor + (obj_dim / 2)
        
        new_loc = list(obj.location)
        new_loc[axis_index] = new_pos
        Library.Trans_Location(obj, new_loc)
        
        cursor = new_pos + (obj_dim / 2) + gap
        
    print(f"Stacked {len(selected_objects)} objects along the {axis}-axis.")

def Space_Selected_Objects(axis='X'):
    """
    Distributes selected objects evenly between the two outermost objects along an axis.
    Parameters: axis (str) - The distribution axis ('X', 'Y', or 'Z').
    Returns: None.
    Example Usage: >>> Space_Selected_Objects(axis='Y') # Evenly spaces objects on the Y-axis
    """
    selected_objects = bpy.context.selected_objects
    if len(selected_objects) < 3:
        print("Warning: At least three objects must be selected to distribute.")
        return

    axis_map = {'X': 0, 'Y': 1, 'Z': 2}
    axis_index = axis_map[axis.upper()]

    sorted_objects = sorted(selected_objects, key=lambda obj: obj.location[axis_index])
    
    start_obj = sorted_objects[0]
    end_obj = sorted_objects[-1]
    middle_objs = sorted_objects[1:-1]
    
    start_pos = start_obj.location[axis_index]
    end_pos = end_obj.location[axis_index]
    
    total_distance = end_pos - start_pos
    num_gaps = len(selected_objects) - 1
    
    if num_gaps == 0: return
    
    increment = total_distance / num_gaps

    for i, obj in enumerate(middle_objs):
        new_pos = start_pos + increment * (i + 1)
        new_loc = list(obj.location)
        new_loc[axis_index] = new_pos
        Library.Trans_Location(obj, new_loc)
        
    print(f"Evenly spaced {len(middle_objs)} objects along the {axis}-axis.")

##########################################################################
# Theme: Plank Construction
##########################################################################

PLANK_LENGTH = 3.0
PLANK_WIDTH = 0.6
PLANK_THICKNESS = 0.2

CHARACTER_BLUEPRINTS = {
    'A': [((-0.8, 0, 1.5), 15), ((0.8, 0, 1.5), -15), ((0, 0, 1.0), 90)],
    'B': [((-1.2, 0, 1.5), 0), ((0.3, 0, 2.8), 90), ((0.3, 0, 1.5), 90), ((0.3, 0, 0.2), 90), ((-1.2, 0, 1.5), 0)],
    'C': [((-1.2, 0, 1.5), 0), ((0.3, 0, 2.8), 90), ((0.3, 0, 0.2), 90)],
    'E': [((-1.0, 0, 1.5), 0), ((0.5, 0, 2.8), 90), ((0.5, 0, 1.5), 90), ((0.5, 0, 0.2), 90)],
    'F': [((-1.0, 0, 1.5), 0), ((0.5, 0, 2.8), 90), ((0.5, 0, 1.5), 90)],
    'H': [((-1.2, 0, 1.5), 0), ((1.2, 0, 1.5), 0), ((0, 0, 1.5), 90)],
    'I': [((0, 0, 1.5), 0)],
    'L': [((-1.0, 0, 1.5), 0), ((0.5, 0, 0.2), 90)],
    'O': [((-1.2, 0, 1.5), 0), ((1.2, 0, 1.5), 0), ((0, 0, 2.8), 90), ((0, 0, 0.2), 90)],
    'T': [((0, 0, 1.5), 0), ((0, 0, 2.8), 90)],
    'U': [((-1.2, 0, 1.5), 0), ((1.2, 0, 1.5), 0), ((0, 0, 0.2), 90)],
    'X': [((0, 0, 1.5), 45), ((0, 0, 1.5), -45)],
    'Y': [((0, 0, 0.8), 0), ((-0.8, 0, 2.2), 30), ((0.8, 0, 2.2), -30)],
    'Z': [((0, 0, 2.8), 90), ((0, 0, 0.2), 90), ((0, 0, 1.5), -65)],
    '1': [((0, 0, 1.5), 0), ((-0.5, 0, 2.5), 45)],
}


def _create_plank_template():
    """Internal helper to create a single, reusable plank object."""
    scale_x = PLANK_LENGTH / 2.0
    scale_y = PLANK_WIDTH / 2.0
    scale_z = PLANK_THICKNESS / 2.0
    
    bpy.ops.mesh.primitive_cube_add(location=(0, 0, -1000))
    template = bpy.context.active_object
    Library.Trans_Scale(template, (scale_x, scale_y, scale_z))
    Library.Apply_Scale(template)
    return template

def _place_plank(template_obj, location, rotation_z_degrees):
    """Internal helper to copy, place, and rotate a plank."""
    new_plank = Library.Copy_Object(template_obj)
    Library.Trans_Location(new_plank, location)
    rotation_z_radians = math.radians(rotation_z_degrees)
    Library.Trans_Rotation(new_plank, (0, 0, rotation_z_radians))
    return new_plank


def Build_From_Text(text, start_location=(0, 0, 0), kerning=3.5, plank_color=None):
    """
    Builds a word or phrase from procedurally placed plank objects.
    Parameters: text (str), start_location (tuple), kerning (float), plank_color (tuple, optional).
    Returns: list[bpy.types.Object] - A list of all the plank objects created.
    Example Usage: >>> Build_From_Text("HELLO", start_location=(0, 5, 0), plank_color=(0.8, 0.5, 0.2))
    """
    template = _create_plank_template()
    all_planks = []
    
    cursor = Vector(start_location)
    
    for char in text.upper():
        if char in CHARACTER_BLUEPRINTS:
            blueprint = CHARACTER_BLUEPRINTS[char]
            for pos_offset, rot_z in blueprint:
                final_pos = cursor + Vector(pos_offset)
                plank = _place_plank(template, final_pos, rot_z)
                if plank_color is not None:
                    Set_Object_Color(plank, plank_color)
                all_planks.append(plank)
        
        cursor.x += kerning
        
    Library.Deselect_All()
    Library.Select_Object(template)
    Library.delete_selected_objects()
    
    print(f"Built '{text}' from {len(all_planks)} planks.")
    return all_planks


def Build_Text_Tower(text, num_layers=5, start_location=(0, 0, 0), color1=(0.8, 0.5, 0.2), color2=(0.6, 0.4, 0.1)):
    """
    Builds a multi-layered tower of text, stacking each layer.
    Parameters: text (str), num_layers (int), start_location (tuple), color1/color2 (tuple).
    Returns: list[bpy.types.Object] - A list of all planks created in the tower.
    Example Usage: >>> Build_Text_Tower("STACK", num_layers=10)
    """
    all_planks = []
    base_cursor = Vector(start_location)
    
    for i in range(num_layers):
        layer_location = base_cursor + Vector((0, 0, i * PLANK_THICKNESS))
        current_color = color1 if i % 2 == 0 else color2
        
        planks_in_layer = Build_From_Text(
            text,
            start_location=layer_location,
            plank_color=current_color
        )
        all_planks.extend(planks_in_layer)
        
    print(f"Built a {num_layers}-layer tower of '{text}' from {len(all_planks)} planks.")
    return all_planks